<?php
/**
 * Theme Functions
 * 
 * This file contains all the theme setup and functionality
 */

function thebizlaunch_custom_fonts() {
    wp_enqueue_style( 'luckiest-guy-font', 'https://fonts.googleapis.com/css2?family=Luckiest+Guy&display=swap' );
}
add_action( 'wp_enqueue_scripts', 'thebizlaunch_custom_fonts' );
// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Theme Setup
 */
function theme_setup() {
    // Add theme support for custom logo
    add_theme_support('custom-logo', array(
        'height' => 100,
        'width' => 400,
        'flex-height' => true,
        'flex-width' => true,
        'header-text' => array('site-title', 'site-description'),
    ));

    // Add theme support for post thumbnails
    add_theme_support('post-thumbnails');

    // Add theme support for HTML5 markup
    add_theme_support('html5', array(
        'search-form',
        'comment-form',
        'comment-list',
        'gallery',
        'caption',
    ));

    // Add theme support for title tag
    add_theme_support('title-tag');

    // Add theme support for custom backgrounds
    add_theme_support('custom-background', array(
        'default-color' => 'ffffff',
    ));

    // Add theme support for selective refresh for widgets
    add_theme_support('customize-selective-refresh-widgets');
}
add_action('after_setup_theme', 'theme_setup');

/**
 * Register navigation menus
 */
function register_theme_menus() {
    register_nav_menus(array(
        'primary' => __('Primary Menu', 'The-Biz-Launch'),
        'mobile' => __('Mobile Menu', 'The-Biz-Launch'),
        'footer' => __('Footer Menu', 'The-Biz-Launch'),
    ));
}
add_action('after_setup_theme', 'register_theme_menus');

/**
 * Enqueue theme styles and scripts
 */
function theme_scripts() {
    // Enqueue theme stylesheet
    wp_enqueue_style('theme-style', get_stylesheet_uri(), array(), wp_get_theme()->get('Version'));
    
    // Enqueue Google Fonts (optional)
    wp_enqueue_style('theme-fonts', 'https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap', array(), null);
    
    // Enqueue theme JavaScript
    wp_enqueue_script('theme-js', get_template_directory_uri() . '/js/theme.js', array('jquery'), wp_get_theme()->get('Version'), true);
    
    // Pass WordPress data to JavaScript
    wp_localize_script('theme-js', 'theme_ajax', array(
        'ajax_url' => admin_url('admin-ajax.php'),
        'nonce' => wp_create_nonce('theme_nonce'),
        'home_url' => home_url('/'),
        'theme_url' => get_template_directory_uri(),
    ));

    // Load comment reply script if needed
    if (is_singular() && comments_open() && get_option('thread_comments')) {
        wp_enqueue_script('comment-reply');
    }
}
add_action('wp_enqueue_scripts', 'theme_scripts');

/**
 * Register widget areas
 */
function theme_widgets_init() {
    register_sidebar(array(
        'name' => __('Primary Sidebar', 'The-Biz-Launch'),
        'id' => 'primary-sidebar',
        'description' => __('Add widgets here to appear in your primary sidebar.', 'The-Biz-Launch'),
        'before_widget' => '<section id="%1$s" class="widget %2$s">',
        'after_widget' => '</section>',
        'before_title' => '<h3 class="widget-title">',
        'after_title' => '</h3>',
    ));

    register_sidebar(array(
        'name' => __('Footer Widget Area', 'The-Biz-Launch'),
        'id' => 'footer-widgets',
        'description' => __('Add widgets here to appear in your footer.', 'The-Biz-Launch'),
        'before_widget' => '<div id="%1$s" class="footer-widget %2$s">',
        'after_widget' => '</div>',
        'before_title' => '<h4 class="footer-widget-title">',
        'after_title' => '</h4>',
    ));
}
add_action('widgets_init', 'theme_widgets_init');

/**
 * Custom Walker Class for Navigation Menus
 */
class Custom_Walker_Nav_Menu extends Walker_Nav_Menu {
    
    /**
     * Starts the list before the elements are added.
     */
    public function start_lvl(&$output, $depth = 0, $args = null) {
        $indent = str_repeat("\t", $depth);
        $output .= "\n$indent<ul class=\"sub-menu\">\n";
    }

    /**
     * Ends the list after the elements are added.
     */
    public function end_lvl(&$output, $depth = 0, $args = null) {
        $indent = str_repeat("\t", $depth);
        $output .= "$indent</ul>\n";
    }

    /**
     * Starts the element output.
     */
    public function start_el(&$output, $item, $depth = 0, $args = null, $id = 0) {
        $indent = ($depth) ? str_repeat("\t", $depth) : '';

        $classes = empty($item->classes) ? array() : (array) $item->classes;
        $classes[] = 'menu-item-' . $item->ID;

        // Add active class for current menu items
        if (in_array('current-menu-item', $classes) || in_array('current-menu-parent', $classes) || in_array('current-menu-ancestor', $classes)) {
            $classes[] = 'active';
        }

        $class_names = join(' ', apply_filters('nav_menu_css_class', array_filter($classes), $item, $args));
        $class_names = $class_names ? ' class="' . esc_attr($class_names) . '"' : '';

        $id = apply_filters('nav_menu_item_id', 'menu-item-' . $item->ID, $item, $args);
        $id = $id ? ' id="' . esc_attr($id) . '"' : '';

        $output .= $indent . '<li' . $id . $class_names .'>';

        $attributes = ! empty($item->attr_title) ? ' title="'  . esc_attr($item->attr_title) .'"' : '';
        $attributes .= ! empty($item->target) ? ' target="' . esc_attr($item->target) .'"' : '';
        $attributes .= ! empty($item->xfn) ? ' rel="'    . esc_attr($item->xfn) .'"' : '';
        $attributes .= ! empty($item->url) ? ' href="'   . esc_attr($item->url) .'"' : '';

        // Add active class to link if current page
        $link_class = '';
        if (in_array('current-menu-item', $classes) || in_array('current-menu-parent', $classes) || in_array('current-menu-ancestor', $classes)) {
            $link_class = ' class="active"';
        }

        $item_output = isset($args->before) ? $args->before : '';
        $item_output .= '<a' . $attributes . $link_class . '>';
        $item_output .= (isset($args->link_before) ? $args->link_before : '') . apply_filters('the_title', $item->title, $item->ID) . (isset($args->link_after) ? $args->link_after : '');
        $item_output .= '</a>';
        $item_output .= isset($args->after) ? $args->after : '';

        $output .= apply_filters('walker_nav_menu_start_el', $item_output, $item, $depth, $args);
    }

    /**
     * Ends the element output.
     */
    public function end_el(&$output, $item, $depth = 0, $args = null) {
        $output .= "</li>\n";
    }
}

/**
 * Custom excerpt length
 */
function custom_excerpt_length($length) {
    return 20;
}
add_filter('excerpt_length', 'custom_excerpt_length', 999);

/**
 * Custom excerpt more text
 */
function custom_excerpt_more($more) {
    return '...';
}
add_filter('excerpt_more', 'custom_excerpt_more');

/**
 * Add custom body classes
 */
function custom_body_classes($classes) {
    // Add a class if it's a mobile device (optional)
    if (wp_is_mobile()) {
        $classes[] = 'mobile-device';
    }

    // Add class for logged-in users
    if (is_user_logged_in()) {
        $classes[] = 'logged-in-user';
    }

    return $classes;
}
add_filter('body_class', 'custom_body_classes');

/**
 * Remove WordPress version from head for security
 */
remove_action('wp_head', 'wp_generator');

/**
 * Disable WordPress emoji scripts for performance
 */
function disable_wp_emojis() {
    remove_action('wp_head', 'print_emoji_detection_script', 7);
    remove_action('admin_print_scripts', 'print_emoji_detection_script');
    remove_action('wp_print_styles', 'print_emoji_styles');
    remove_action('admin_print_styles', 'print_emoji_styles');
    remove_filter('the_content_feed', 'wp_staticize_emoji');
    remove_filter('comment_text_rss', 'wp_staticize_emoji');
    remove_filter('wp_mail', 'wp_staticize_emoji_for_email');
}
add_action('init', 'disable_wp_emojis');

/**
 * Add theme customizer options
 */
function theme_customize_register($wp_customize) {
    // Add a section for theme colors
    $wp_customize->add_section('theme_colors', array(
        'title' => __('Theme Colors', 'The-Biz-Launch'),
        'priority' => 30,
    ));

    // Primary color setting
    $wp_customize->add_setting('primary_color', array(
        'default' => '#f77878',
        'sanitize_callback' => 'sanitize_hex_color',
    ));

    $wp_customize->add_control(new WP_Customize_Color_Control($wp_customize, 'primary_color', array(
        'label' => __('Primary Color', 'The-Biz-Launch'),
        'section' => 'theme_colors',
        'settings' => 'primary_color',
    )));

    // Secondary color setting
    $wp_customize->add_setting('secondary_color', array(
        'default' => '#764ba2',
        'sanitize_callback' => 'sanitize_hex_color',
    ));

    $wp_customize->add_control(new WP_Customize_Color_Control($wp_customize, 'secondary_color', array(
        'label' => __('Secondary Color', 'The-Biz-Launch'),
        'section' => 'theme_colors',
        'settings' => 'secondary_color',
    )));
}
add_action('customize_register', 'theme_customize_register');

/**
 * Output custom colors from customizer
 */
function theme_custom_colors() {
    $primary_color = get_theme_mod('primary_color', '#f77878');
    $secondary_color = get_theme_mod('secondary_color', '#764ba2');
    
    echo '<style type="text/css">';
    echo ':root { --primary-color: ' . esc_attr($primary_color) . '; --secondary-color: ' . esc_attr($secondary_color) . '; }';
    echo '</style>';
}
add_action('wp_head', 'theme_custom_colors');

/**
 * Add security headers
 */
function add_security_headers() {
    if (!is_admin()) {
        header('X-Content-Type-Options: nosniff');
        header('X-Frame-Options: SAMEORIGIN');
        header('X-XSS-Protection: 1; mode=block');
        header('Referrer-Policy: strict-origin-when-cross-origin');
    }
}
add_action('send_headers', 'add_security_headers');

/**
 * Optimize WordPress for performance
 */
function optimize_wordpress() {
    // Remove unnecessary meta tags
    remove_action('wp_head', 'rsd_link');
    remove_action('wp_head', 'wlwmanifest_link');
    remove_action('wp_head', 'start_post_rel_link', 10, 0);
    remove_action('wp_head', 'parent_post_rel_link', 10, 0);
    remove_action('wp_head', 'adjacent_posts_rel_link', 10, 0);
}
add_action('init', 'optimize_wordpress');
?>