(function($) {
    'use strict';

    // Mobile menu toggle
    function toggleMobileMenu() {
        const mobileNav = document.getElementById('mobileNav');
        const mobileToggle = document.querySelector('.mobile-toggle');
       
        if (mobileNav && mobileToggle) {
            mobileNav.classList.toggle('active');
            mobileToggle.classList.toggle('active');
        }
    }

    // Dark mode toggle
    function toggleDarkMode() {
        const body = document.body;
        const currentTheme = body.getAttribute('data-theme');
       
        if (currentTheme === 'dark') {
            body.setAttribute('data-theme', 'light');
            localStorage.setItem('theme', 'light');
            updateDarkModeButtons('🌙 Dark Mode');
        } else {
            body.setAttribute('data-theme', 'dark');
            localStorage.setItem('theme', 'dark');
            updateDarkModeButtons('☀️ Light Mode');
        }
    }

    // Update dark mode button text
    function updateDarkModeButtons(text) {
        const desktopBtn = document.querySelector('.dark-mode-toggle');
        const mobileBtn = document.querySelector('.mobile-dark-toggle');
       
        if (desktopBtn) {
            desktopBtn.innerHTML = text;
        }
        if (mobileBtn) {
            mobileBtn.innerHTML = text;
        }
    }

    // Handle WordPress menu active states
    function setActiveMenuItem() {
        const currentUrl = window.location.pathname;
        const menuLinks = document.querySelectorAll('.nav-menu a, .mobile-nav a');
        
        menuLinks.forEach(link => {
            const linkUrl = new URL(link.href).pathname;
            
            // Remove existing active classes
            link.classList.remove('active');
            
            // Add active class if URLs match
            if (linkUrl === currentUrl || (currentUrl === '/' && linkUrl === '/')) {
                link.classList.add('active');
            }
        });
    }

    // Handle menu link clicks for SPA-like behavior (optional)
    function handleMenuClicks() {
        document.querySelectorAll('.mobile-nav a').forEach(link => {
            link.addEventListener('click', function() {
                // Close mobile menu when link is clicked
                const mobileNav = document.getElementById('mobileNav');
                const mobileToggle = document.querySelector('.mobile-toggle');
                
                if (mobileNav && mobileNav.classList.contains('active')) {
                    mobileNav.classList.remove('active');
                    mobileToggle.classList.remove('active');
                }
            });
        });
    }

    // Close mobile menu when clicking outside
    function handleOutsideClick() {
        document.addEventListener('click', function(event) {
            const mobileNav = document.getElementById('mobileNav');
            const mobileToggle = document.querySelector('.mobile-toggle');
            const navigation = document.querySelector('.navigation');
           
            if (mobileNav && mobileToggle && navigation) {
                if (!navigation.contains(event.target) && mobileNav.classList.contains('active')) {
                    mobileNav.classList.remove('active');
                    mobileToggle.classList.remove('active');
                }
            }
        });
    }

    // Close mobile menu on window resize
    function handleWindowResize() {
        window.addEventListener('resize', function() {
            if (window.innerWidth > 768) {
                const mobileNav = document.getElementById('mobileNav');
                const mobileToggle = document.querySelector('.mobile-toggle');
                
                if (mobileNav && mobileToggle) {
                    mobileNav.classList.remove('active');
                    mobileToggle.classList.remove('active');
                }
            }
        });
    }

    // Initialize dark mode from localStorage
    function initializeDarkMode() {
        const savedTheme = localStorage.getItem('theme') || 'light';
        document.body.setAttribute('data-theme', savedTheme);
       
        if (savedTheme === 'dark') {
            updateDarkModeButtons('☀️ Light Mode');
        } else {
            updateDarkModeButtons('🌙 Dark Mode');
        }
    }

    // Add event listeners for dark mode toggle buttons
    function addDarkModeListeners() {
        const desktopBtn = document.querySelector('.dark-mode-toggle');
        const mobileBtn = document.querySelector('.mobile-dark-toggle');
        
        if (desktopBtn) {
            desktopBtn.addEventListener('click', function(e) {
                e.preventDefault();
                toggleDarkMode();
            });
        }
        
        if (mobileBtn) {
            mobileBtn.addEventListener('click', function(e) {
                e.preventDefault();
                toggleDarkMode();
            });
        }
    }

    // Debug function to check if elements exist
    function debugElements() {
        console.log('Dark mode toggle elements:');
        console.log('Desktop button:', document.querySelector('.dark-mode-toggle'));
        console.log('Mobile button:', document.querySelector('.mobile-dark-toggle'));
        console.log('Body data-theme:', document.body.getAttribute('data-theme'));
        console.log('Saved theme:', localStorage.getItem('theme'));
    }

    // Make functions globally available for onclick handlers IMMEDIATELY
    window.toggleMobileMenu = toggleMobileMenu;
    window.toggleDarkMode = toggleDarkMode;

    // Initialize dark mode IMMEDIATELY (before DOM ready)
    initializeDarkMode();

    // Initialize everything when DOM is ready
    $(document).ready(function() {
        // Debug - remove this line after testing
        debugElements();
        
        // Re-initialize to make sure everything is set correctly
        initializeDarkMode();
        setActiveMenuItem();
        handleMenuClicks();
        handleOutsideClick();
        handleWindowResize();

        // Handle AJAX navigation if using WordPress AJAX
        $(document).on('wp_loaded', function() {
            setActiveMenuItem();
        });
    });

    // Re-initialize on AJAX page loads (if using AJAX navigation)
    $(document).ajaxComplete(function() {
        setActiveMenuItem();
        addDarkModeListeners(); // Re-add listeners after AJAX loads
    });

})(jQuery);